<?php
namespace App\Http\Controllers;
use OpenApi\Annotations as OA;
use Illuminate\Http\Request;
use App\Models\Project;
use App\Models\Organization;
use App\Models\ProjectResult;
use App\Models\ProjectObjective;
use App\Models\ProjectThematic;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Models\Document;
use Illuminate\Support\Str;
/**
 * @OA\Tag(
 *     name="Projects",
 *     description="Endpoints related to project management, such as creating, listing, and deleting projects."
 * )
*/
class ProjectController extends Controller
{
    /**
     * @OA\Get(
     *     path="/api/projects",
     *     summary="Get a list of projects",
     *     tags={"Projects"},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function index()
    {
        //$projects = Project::all();
        $projects = DB::table('tabular.projects')
        ->leftJoin('tabular.organizations', 'tabular.organizations.id', '=', 'tabular.projects.organization_id')
        ->leftJoin('tabular.project_thematics', 'tabular.project_thematics.project_id', '=', 'tabular.projects.id')
        ->select('tabular.projects.*',
            'tabular.organizations.logo as organization_logo_path',
            DB::raw("string_agg(tabular.project_thematics.thematic_id::text, ',') as thematic_ids")
        )
        ->groupBy('tabular.projects.id', 'tabular.organizations.logo')
        ->get();
        return response()->json([
            'status' => 'success',
            "date_now" => Carbon::now()->format('Y-m-d'),
            'data' => $projects,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Get(
     *     path="/api/project/{id}",
     *     summary="Get project info from ID",
     *     tags={"Projects"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Project ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Project informations",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="field", type="string"),
     *                 @OA\Property(property="start_date", type="string"),
     *                 @OA\Property(property="end_date", type="string"),
     *                 @OA\Property(property="currency_id", type="string"),
     *                 @OA\Property(property="user_id", type="string"),
     *                 @OA\Property(property="website", type="string"),
     *                 @OA\Property(property="manager_name", type="string"),
     *                 @OA\Property(property="manager_phone", type="string"),
     *                 @OA\Property(property="focal_point_name", type="string"),
     *                 @OA\Property(property="focal_point_email", type="string"),
     *                 @OA\Property(property="focal_point_phone", type="string"),
     *                 @OA\Property(property="main_objective", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string"),
     *                 @OA\Property(property="location", type="object",
     *                      @OA\Property(property="id", type="string")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function info(Request $request)
    {
        $data = DB::table('tabular.projects')
        ->leftJoin('codelist.project_locations', 'tabular.projects.location_id', '=', 'project_locations.id')
        ->leftJoin('codelist.currency', 'tabular.projects.currency_id', '=', 'codelist.currency.id')
        ->leftJoin(DB::raw("
            (
                SELECT d1.path, d1.project_id
                FROM tabular.documents d1
                WHERE d1.type LIKE 'image%'
                GROUP BY d1.project_id, d1.path
            ) as first_images
        "), 'tabular.projects.id', '=', 'first_images.project_id')
        ->leftJoin('tabular.organizations', 'tabular.organizations.id', '=', 'tabular.projects.organization_id')
        ->select('tabular.projects.*',
            'project_locations.name as location_name',
            'first_images.path as image_path',
            'currency.name as currency',
            'tabular.organizations.logo as organization_logo_path',
        )
        ->where('tabular.projects.id', $request->id)
        ->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }


    /**
     * @OA\Post(
     *     path="/api/project",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new project",
     *     @OA\RequestBody(
     *          required=true,
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *              required={"name", "user_id", "organization_id"},
     *                    @OA\Property(property="name", type="string", example="Project name"),
     *                    @OA\Property(
     *                     property="logo",
     *                     description="Logo file to upload",
     *                     type="string",
     *                     format="binary"
     *                    ),
     *                    @OA\Property(property="field", type="string", example="Project field"),
     *                    @OA\Property(property="location_id", type="string", example="Location ID"),
     *                    @OA\Property(property="start_date", type="date"),
     *                   @OA\Property(property="end_date", type="date"),
     *                   @OA\Property(property="budget", type="integer"),
     *                   @OA\Property(property="currency_id", type="string"),
     *                   @OA\Property(property="organization_id", type="string"),
     *                   @OA\Property(property="user_id", type="string"),
     *                   @OA\Property(property="website", type="string"),
     *                   @OA\Property(property="manager_name", type="string"),
     *                   @OA\Property(property="manager_email", type="string"),
     *                   @OA\Property(property="manager_phone", type="string"),
     *                   @OA\Property(property="focal_point_name", type="string"),
     *                   @OA\Property(property="focal_point_email", type="string"),
     *                   @OA\Property(property="focal_point_phone", type="string"),
     *                   @OA\Property(property="main_objective", type="string")
     *                 )
     *            )
     *     ),
     *
     *
     *     @OA\Response(
     *         response=200,
     *         description="Account created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="field", type="string"),
     *                 @OA\Property(property="start_date", type="string"),
     *                 @OA\Property(property="end_date", type="string"),
     *                 @OA\Property(property="currency_id", type="string"),
     *                 @OA\Property(property="user_id", type="string"),
     *                 @OA\Property(property="website", type="string"),
     *                 @OA\Property(property="manager_name", type="string"),
     *                 @OA\Property(property="manager_phone", type="string"),
     *                 @OA\Property(property="focal_point_name", type="string"),
     *                 @OA\Property(property="focal_point_email", type="string"),
     *                 @OA\Property(property="focal_point_phone", type="string"),
     *                 @OA\Property(property="main_objective", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string"),
     *                 @OA\Property(property="location", type="object",
     *                      @OA\Property(property="id", type="string")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'organization_id' => 'required|string',
            'logo' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'user_id' => 'required|string',
            'start_date' => 'date', // Format Y-m-d, etc.
            'end_date' => 'date', // Format Y-m-d, etc.
        ]);
        $dateDebut = Carbon::parse($request->date_debut);
        $dateFin = Carbon::parse($request->date_fin);
        if ($dateFin->lt($dateDebut)) {
            return response()->json(['error' => 'The end date must be after the start date.'], 400);
        }
        //Check if user is editor or adminisrator
        $path = null;
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $imageName = Str::uuid()->toString().'.'.$file->extension();
            $path = $file->storeAs('logos', $imageName, 'public');
            // Check if file has been uploaded
            if (!$path) {
                return response()->json([
                    'message' => 'Logo failed to upload.',
                ], 500);
            }
        }
        $data = Project::create([
            'name' => $request->name,
            'logo' => $path,
            'organization_id' => $request->organization_id,
            'field' => $request->field,
            'location_id' => $request->location_id,
            'user_id' => $request->user_id,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'budget' => $request->budget,
            'currency_id' => $request->currency_id,
            'organization_id' => $request->organization_id,
            'manager_email' => $request->manager_email,
            'manager_name' => $request->manager_name,
            'manager_phone' => $request->manager_phone,
            'focal_point_name' => $request->focal_point_name,
            'focal_point_email' => $request->focal_point_email,
            'focal_point_phone' => $request->focal_point_phone,
            'main_objective' => $request->main_objective
        ]);
        return response()->json([
            'message' => 'Project created successfully',
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Patch(
     *     path="/api/project/{id}",
     *     summary="Update an existing project",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Project ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name", "user_id", "organization_id"},
     *             @OA\Property(property="name", type="string", example="Project name"),
     *             @OA\Property(property="field", type="string", example="Project field"),
     *             @OA\Property(property="location_id", type="string", example="Location ID"),
     *             @OA\Property(property="start_date", type="date"),
     *             @OA\Property(property="end_date", type="date"),
     *             @OA\Property(property="budget", type="integer"),
     *             @OA\Property(property="currency_id", type="string"),
     *             @OA\Property(property="organization_id", type="string"),
     *             @OA\Property(property="website", type="string"),
     *             @OA\Property(property="manager_name", type="string"),
     *             @OA\Property(property="manager_email", type="string"),
     *             @OA\Property(property="manager_phone", type="string"),
     *             @OA\Property(property="focal_point_name", type="string"),
     *             @OA\Property(property="focal_point_email", type="string"),
     *             @OA\Property(property="focal_point_phone", type="string"),
     *             @OA\Property(property="main_objective", type="string")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Project updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="field", type="string"),
     *                 @OA\Property(property="start_date", type="string"),
     *                 @OA\Property(property="end_date", type="string"),
     *                 @OA\Property(property="currency_id", type="string"),
     *                 @OA\Property(property="user_id", type="string"),
     *                 @OA\Property(property="website", type="string"),
     *                 @OA\Property(property="manager_name", type="string"),
     *                 @OA\Property(property="manager_phone", type="string"),
     *                 @OA\Property(property="focal_point_name", type="string"),
     *                 @OA\Property(property="focal_point_email", type="string"),
     *                 @OA\Property(property="focal_point_phone", type="string"),
     *                 @OA\Property(property="main_objective", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string"),
     *                 @OA\Property(property="location", type="object",
    *                      @OA\Property(property="id", type="string")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(response=404, description="Project not found"),
     *     @OA\Response(response=401, description="Validation error")
     * )
     */
    public function update(Request $request, $id)
    {
        // Validate only present fields
        $validated = $request->validate([
            'name' => 'required|string',
            'logo' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'organization_id' => 'required|string',
            'start_date' => 'date', // Format Y-m-d, etc.
            'end_date' => 'date', // Format Y-m-d, etc.
        ]);
        $dateDebut = Carbon::parse($request->date_debut);
        $dateFin = Carbon::parse($request->date_fin);
        if ($dateFin->lt($dateDebut)) {
            return response()->json(['error' => 'The end date must be after the start date.'], 400);
        }
        $path = null;
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $imageName = Str::uuid()->toString().'.'.$file->extension();
            $path = $file->storeAs('logos', $imageName, 'public');
        }
        // Check if file has been uploaded
        if (!$path) {
            return response()->json([
                'message' => 'Logo failed to upload.',
            ], 500);
        }
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        // Find the project or return 404
        $project = Project::findOrFail($id);
        // Update project with validated data
        if($path) {
           // $validated["logo"] = $path;
        }
        $project->update($validated);
        return response()->json([
            'status' => 'success',
            'message' => 'Project updated successfully.',
            'data' => $project
        ]);
    }

    /**
     * @OA\Delete(
     *     path="/api/project/{id}",
     *     summary="Delete a project",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Project ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Project deleted successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="string", example="success"),
     *             @OA\Property(property="message", type="string", example="Project deleted successfully.")
     *         )
     *     ),
     *     @OA\Response(response=404, description="Project not found")
     * )
     */
    public function delete($id)
    {
        $project = Project::find($id);
        //Check if user has role to delete this project
        if (!$project) {
            return response()->json([
                'status' => 'error',
                'message' => 'Project not found.'
            ], 404);
        }
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        $project->delete();
        return response()->json([
            'status' => 'success',
            'message' => 'Project deleted successfully.'
        ]);
    }

    /**
     * @OA\Post(
     *     path="/api/project/objectives",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new project objective",
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name"},
     *             @OA\Property(property="name", type="string", example="Project result"),
     *             @OA\Property(property="project_id", type="string", example="Project ID")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Objective created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="project_id", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function objective(Request $request)
    {
        $request->validate([
            'name' => 'required|string'
        ]);
        $data = ProjectObjective::create([
            'name' => $request->name,
            'project_id' => $request->project_id
        ]);
        return response()->json([
            'message' => 'Objective created successfully',
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Post(
     *     path="/api/project/results",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new project result",
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name"},
     *             @OA\Property(property="name", type="string", example="Project objective"),
     *             @OA\Property(property="project_id", type="string", example="Project ID")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Result created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="project_id", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function result(Request $request)
    {
        $request->validate([
            'name' => 'required|string'
        ]);
        $data = ProjectResult::create([
            'name' => $request->name,
            'project_id' => $request->project_id
        ]);
        return response()->json([
            'message' => 'Result created successfully',
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Get(
     *     path="/api/project/objectives",
     *     summary="Get a list of project objectives",
     *     @OA\Parameter(
     *         required=true,
     *         name="project_id",
     *         in="query",
     *         description="UUID or ID of the project",
     *         @OA\Schema(type="string")
     *     ),
     *     tags={"Projects"},
     *     @OA\Response(
*             response=200,
*             description="Success",
    *         @OA\JsonContent(
    *             @OA\Property(property="data", type="array", @OA\Items(type="object"))
    *         )
     *      )
     * )
     */
    public function getObjectives(Request $request)
    {
        $request->validate([
            'project_id' => 'required|string'
        ]);
        $data = ProjectObjective::where('project_id', $request->project_id)->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Get(
     *     path="/api/project/results",
     *     summary="Get a list of project results",
     *     @OA\Parameter(
     *         required=true,
     *         name="project_id",
     *         in="query",
     *         description="UUID or ID of the project",
     *         @OA\Schema(type="string")
     *     ),
     *     tags={"Projects"},
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\JsonContent(
     *             @OA\Property(property="data", type="array", @OA\Items(type="object"))
     *         )
     *      )
     * )
    */
    public function getResults(Request $request)
    {
        $request->validate([
            'project_id' => 'required|string'
        ]);
        $data = ProjectResult::where('project_id', $request->project_id)->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Post(
     *     path="/api/project/thematics",
     *     tags={"Projects"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new project thematic",
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"project_id", "thematic_id"},
     *             @OA\Property(property="data_licence_id", type="string", example="Data licence ID"),
     *             @OA\Property(
     *                 property="collection_method_ids",
     *                 type="array",
     *                 @OA\Items(type="string", format="uuid", example="550e8400-e29b-41d4-a716-446655440000")
     *             ),
     *             @OA\Property(
     *                 property="data_format_ids",
     *                 type="array",
     *                 @OA\Items(type="string", format="uuid", example="550e8400-e29b-41d4-a716-446655440000")
     *             ),
     *             @OA\Property(property="project_id", type="string", example="Project ID"),
     *             @OA\Property(property="thematic_id", type="string", example="Thematic ID")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Project thematic created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string", example="Project thematic created successfully"),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="id", type="string", example="uuid"),
     *                 @OA\Property(property="data_licence_id", type="string", example="Data licence ID"),
     *                 @OA\Property(
     *                     property="collection_method_ids",
     *                     type="array",
     *                     @OA\Items(type="string", format="uuid", example="550e8400-e29b-41d4-a716-446655440000")
     *                 ),
     *                 @OA\Property(
     *                     property="data_format_ids",
     *                     type="array",
     *                     @OA\Items(type="string", format="uuid", example="550e8400-e29b-41d4-a716-446655440000")
     *                 ),
     *                 @OA\Property(property="project_id", type="string", example="Project ID"),
     *                 @OA\Property(property="thematic_id", type="string", example="Thematic ID")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function thematic(Request $request)
    {
        $request->validate([
            'project_id' => 'required|string',
            'thematic_id' => 'required|string',
            'collection_method_ids' => ['array'],
            //'collection_method_ids.*' => [Rule::exists(DB::raw('"codelist"."collection_methods"'), 'id')],
            'data_format_ids' => ['array'],
            //'data_format_ids.*' => [Rule::exists(DB::raw('"codelist"."data_formats"'), 'id')],
        ]);

        $data = ProjectThematic::create([
            'data_licence_id' => $request->data_licence_id,
            'collection_method_ids' => $request->collection_method_ids,
            'data_format_ids' => $request->data_format_ids,
            'project_id' => $request->project_id,
            'thematic_id' => $request->thematic_id
        ]);
        return response()->json([
            'message' => 'Project thematic created successfully',
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Get(
     *     path="/api/project/thematics",
     *     summary="Get a list of project thematics",
     *     @OA\Parameter(
     *         required=true,
     *         name="project_id",
     *         in="query",
     *         description="UUID or ID of the project",
     *         @OA\Schema(type="string")
     *     ),
     *     tags={"Projects"},
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\JsonContent(
     *             @OA\Property(property="data", type="array", @OA\Items(type="object"))
     *         )
     *      )
     * )
    */
    public function getThematics(Request $request)
    {
        $request->validate([
            'project_id' => 'required|string'
        ]);
        $data = DB::table('tabular.project_thematics')
        ->leftJoin('tabular.thematics', 'tabular.project_thematics.thematic_id', '=', 'tabular.thematics.id')
        ->leftJoin('codelist.thematic_categories', 'thematics.category_thematic_id', '=', 'thematic_categories.id')
        ->select('tabular.project_thematics.id as thematic_id',
            'codelist.thematic_categories.name as category_name',
            'tabular.thematics.name as name'
        )
        ->where('tabular.project_thematics.project_id', $request->project_id)
        ->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }
}
