<?php

namespace App\Http\Controllers;
use OpenApi\Annotations as OA;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\Organization;
use App\Models\RefreshToken;
/**
 * @OA\Tag(
 *     name="Auth",
 *     description="Endpoints related to authentification management."
 * )
*/
class AuthController extends Controller
{
     /**
     * @OA\Post(
     *     path="/api/login",
     *     tags={"Auth"},
     *     summary="Log in (and return access token)",
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"username","password"},
     *             @OA\Property(property="username", type="string", format="email", example="john@example.com"),
     *             @OA\Property(property="password", type="string", format="password", example="password123")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful login",
     *         @OA\JsonContent(
     *             @OA\Property(
     *                 property="user",
     *                 type="object",
     *                 @OA\Property(property="name", type="string", example="John Doe"),
     *                 @OA\Property(property="username", type="string", example="john@example.com"),
     *             ),
     *              @OA\Property(property="access_token", type="string"),
    *                 @OA\Property(property="refresh_token", type="string"),
    *                 @OA\Property(property="token_type", type="string", example="Bearer"),
    *                 @OA\Property(property="expires_in", type="integer")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Invalid credentials"
     *     )
     * )
     */
    public function login(Request $request)
    {
        $request->validate([
            'username'    => 'required|email',
            'password' => 'required',
        ]);
        $user = User::where('email', $request->username)->first();
        if (! $user || ! Hash::check($request->password, $user->password)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }
        $token = $user->createToken('access-token', ['user_id=' . $user->id, 'role_id=' . $user->role, 'organization_id=' . $user->organization_id])->plainTextToken;
        // Create refresh token
        $refreshToken = hash('sha512', bin2hex(random_bytes(64)));
        $user_role = DB::table('codelist.user_roles')->where('id', $user->role_id)->get()->first();
        $user_organization = Organization::find($user->organization_id);
        RefreshToken::create([
            'user_id' => $user->id,
            'access_token' => $token,
            'refresh_token' => $refreshToken,
            'access_token_expires_at' => now()->addMinutes(60),   // Very short
            'refresh_token_expires_at' => now()->addDays(30),     // Long
        ]);
        return response()->json([
            'user'  => $user->makeHidden(['id']),
            'access_token' =>  $token,
            'refresh_token' => $refreshToken,
            'role_name' => $user_role->name,
            'organization_name' => $user_organization->name,
            'token_type' => 'Bearer',
            'expires_in' => 3600, //60*60 (seconds)
        ]);
    }

    /**
     * @OA\Post(
     *     path="/api/logout",
     *     summary="Log out",
     *     tags={"Auth"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Response(response=200, description="Successful logout")
     * )
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return response()->json(['message' => 'Logged out']);
    }

    public function me(Request $request)
    {
        return response()->json($request->user());
    }

    public function refresh(Request $request)
    {
        $request->validate([
            'refresh_token' => 'required|string'
        ]);
        $hashed = hash('sha512', $request->refresh_token);
        $token = RefreshToken::where('refresh_token', $hashed)->first();
        if (!$token || $token->isExpired()) {
            return response()->json(['message' => 'Invalid or expired refresh token'], 401);
        }
        $user = $token->user;
        // Optionally invalidate old token
        $token->delete();
        // Generate new tokens
        $newAccessToken = $user->createToken('access_token')->plainTextToken;
        $newRefreshToken = bin2hex(random_bytes(64));;
        RefreshToken::create([
            'user_id' => $user->id,
            'access_token' => hash('sha512', $newRefreshToken),
            'refresh_token' => hash('sha512', $newRefreshToken),
            'expires_at' => now()->addDays(30),
        ]);
        return response()->json([
            'access_token' => $newAccessToken,
            'refresh_token' => $newRefreshToken,
            'expires_in' => 3600
        ]);
    }

}
