<?php

namespace App\Http\Controllers;
use App\Models\Organization;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
/**
 * @OA\Tag(
 *     name="Organizations",
 *     description="Endpoints related to organization management, such as creating, listing, and deleting organizations."
 * )
*/
class OrganizationController extends Controller
{
    /**
     * @OA\Get(
     *     path="/api/organizations",
     *     summary="Get a list of organization",
     *     tags={"Organizations"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function index()
    {
        $data = Organization::all();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Post(
     *     path="/api/organization",
     *     tags={"Organizations"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new organizations",
     *     @OA\RequestBody(
     *          required=true,
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                    required={"name"},
     *                    @OA\Property(property="name", type="string", description="Organization name"),
     *                    @OA\Property(
     *                     property="logo",
     *                     description="Logo file to upload",
     *                     type="string",
     *                     format="binary"
     *                    ),
     *                    @OA\Property(property="description", type="text", description="Organization description")
     *                 )
     *            )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Organization created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="logo", type="string"),
     *                 @OA\Property(property="description", type="text"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function store(Request $request)
    {

        $request->validate([
            'name' => 'required|string',
            'logo' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string'
        ]);
        $path = null;
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $imageName = Str::uuid()->toString().'.'.$file->extension();
            $path = $file->storeAs('logos', $imageName, 'public');
            // Check if file has been uploaded
            if (!$path) {
                return response()->json([
                    'message' => 'Logo failed to upload.',
                ], 500);
            }
        }
        $data = Organization::create([
            'name' => $request->name,
            'description' => $request->description,
            'logo' => $path,
        ]);
        return response()->json([
            'message' => 'Organization created successfully',
            'data' => $data,
        ], 200);
    }




    /**
     * @OA\Patch(
     *     path="/api/organization/{id}",
     *     summary="Update an organization's details",
     *     tags={"Organizations"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID of the organization",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"name"},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="New Organization Name"
     *                 ),
     *                 @OA\Property(
     *                     property="logo",
     *                     type="string",
     *                     format="binary",
     *                     description="Logo file to upload"
     *                 ),
     *                 @OA\Property(
     *                     property="description",
     *                     type="string",
     *                     description="New Organization Description"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Organization updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="logo", type="string"),
     *                 @OA\Property(property="description", type="string"),
     *                 @OA\Property(property="created_at", type="date"),
     *                 @OA\Property(property="updated_at", type="date")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */

    public function update(Request $request, $id)
    {

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'logo' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string'
        ]);
        //$organization->name = $request->name;
        //$organization->description = $request->description;
        $path = null;
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $imageName = Str::uuid()->toString().'.'.$file->extension();
            $thumbnailPath = Storage::disk('public')->path('documents/' . $baseDocumentName .'_thumbnail.jpg');
            $path = $file->storeAs('logos', $imageName, 'public');
            $validated->path=$path;
        }
        $data = Organization::findOrFail($id);
        $data->update($validated);
        return response()->json([
            'message' => 'Organization updated successfully',
            'data' => $organization,
        ], 200);
    }
}
