<?php

namespace App\Http\Controllers;
use OpenApi\Annotations as OA;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
/**
 * @OA\Tag(
 *     name="Spatial",
 *     description="Endpoints related to Indicators management (Spatial and not spatial), such as listing, and editing indicator."
 * )
*/
class IndicatorController extends Controller
{
    /**
     * @OA\Patch(
     *     path="/api/spatial/table/{table}/{id}",
     *     summary="Update a spatial data table",
     *     tags={"Spatial"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Spatial data ID",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Parameter(
     *         name="table",
     *         in="path",
     *         required=true,
     *         description="Spatial data table name",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         description="Accepts an unknown JSON object",
     *         @OA\JsonContent(
     *             type="object",
     *             additionalProperties=true,
     *             example={"anyKey1": "anyValue1", "anyKey2": "anyValue2"}
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Spatial data updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=404, description="Data not found"),
     *     @OA\Response(response=401, description="Validation error")
     * )
     */
    public function updateSpatialDataTable(Request $request, $table, $id)
    {
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        $data = $request->json()->all();
        $updated = DB::table($table)->where('id', $id)->update($data);
        if (!$updated) {
            return response()->json(['error' => "Record not found or not updated"], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Table updated successfully.',
            'data' => $data
        ]);
    }

    /**
     * @OA\Get(
     *     path="/api/spatial/legend",
     *     summary="Get a list of spatial legend",
     *     tags={"Spatial"},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function index()
    {
        $data = DB::table('spatial.legend')->orderBy('zindex')->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Post(
     *     path="/api/spatial/legend",
     *     tags={"Spatial"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new legend",
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name", "table", "type", "stroke_color", "stroke_width"},
     *             @OA\Property(property="name", type="string", description="Legend label"),
     *             @OA\Property(property="table", type="string", description="Related legend table"),
     *             @OA\Property(property="layer_name", type="string", description="Layer name"),
     *             @OA\Property(property="zindex", type="integer", description="Layer Z-index"),
     *             @OA\Property(
     *                 property="type",
     *                 type="string",
     *                 description="Data geometry type",
     *                 enum={"POLYGON", "LINE", "POINT"},
     *                 example="POLYGON"
     *             ),
     *             @OA\Property(property="stroke_color", type="string", description="Stroke color"),
     *             @OA\Property(property="stroke_width", type="integer", description="Stroke width"),
     *             @OA\Property(property="fill_color", type="string", description="Fill color, usefull just if POLYGON type has been selected")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Thematic created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="table", type="string"),
     *                 @OA\Property(property="layer_name", type="string"),
     *                 @OA\Property(property="zindex", type="integer"),
     *                 @OA\Property(property="type", type="string"),
     *                 @OA\Property(property="stroke_color", type="string"),
     *                 @OA\Property(property="stroke_width", type="integer"),
     *                 @OA\Property(property="fill_color", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function storeLegend(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'table' => 'required|string',
            'layer_name' => 'required|string',
            'type' => 'required|string',
            'stroke_color' => 'required|string',
            'stroke_width' => 'required|numeric',
            'zindex' => 'numeric'
        ]);
        // Insert data and get the inserted ID
        $id = DB::table('spatial.legend')->insertGetId([
            'name' => $request->name,
            'layer_name' => $request->layer_name,
            'table' => $request->table,
            'type' => $request->type,
            'stroke_color' => $request->stroke_color,
            'stroke_width' => $request->stroke_width,
            'zindex' => $request->zindex,
            'fill_color' => $request->fill_color
        ]);
        // Retrieve the inserted row
        $data = DB::table('spatial.legend')->where('id', $id)->first();
        return response()->json([
            'message' => 'Legend entry created successfully',
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Patch(
     *     path="/api/spatial/legend/{id}",
     *     summary="Update a legend entry",
     *     tags={"Spatial"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Legend ID",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name", "table", "type", "stroke_color", "stroke_width", "layer_name"},
     *             @OA\Property(property="name", type="string", description="Legend label"),
     *             @OA\Property(property="table", type="string", description="Related legend table"),
     *             @OA\Property(property="layer_name", type="string", description="Related legend layer name"),
     *             @OA\Property(property="zindex", type="integer", description="Layer Z-index"),
     *             @OA\Property(
     *                 property="type",
     *                 type="string",
     *                 description="Data geometry type",
     *                 enum={"POLYGON", "LINE", "POINT"},
     *                 example="POLYGON"
     *             ),
     *             @OA\Property(property="stroke_color", type="string", description="Stroke color"),
     *             @OA\Property(property="stroke_width", type="integer", description="Stroke width"),
     *             @OA\Property(property="fill_color", type="string", description="Stroke width, usefull just if POLYGON type has been selected")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Spatial data updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="table", type="string"),
     *                 @OA\Property(property="layer_name", type="string"),
     *                 @OA\Property(property="zindex", type="integer"),
     *                 @OA\Property(property="type", type="string"),
     *                 @OA\Property(property="stroke_color", type="string"),
     *                 @OA\Property(property="stroke_width", type="string"),
     *                 @OA\Property(property="fill_color", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=404, description="Data not found"),
     *     @OA\Response(response=401, description="Validation error")
     * )
     */
    public function updateLegend(Request $request, $id)
    {
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        $data = $request->json()->all();
        $updated = DB::table("spatial.legend")->where('id', $id)->update($data);
        if (!$updated) {
            return response()->json(['error' => "Record not found or not updated"], 404);
        }
        // Retrieve the inserted row
        $data = DB::table('spatial.legend')->where('id', $id)->first();
        return response()->json([
            'status' => 'success',
            'message' => 'Table updated successfully.',
            'data' => $data
        ]);
    }

    /**
     * @OA\Delete(
     *     path="/api/spatial/legend/{id}",
     *     summary="Delete a legend entry",
     *     tags={"Spatial"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Legend ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Legend entry deleted successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="string", example="success"),
     *             @OA\Property(property="message", type="string", example="Legend entry deleted successfully.")
     *         )
     *     ),
     *     @OA\Response(response=404, description="Legend entry not found")
     * )
     */
    public function deleteLegend($id)
    {
        $row = DB::table('spatial.legend')->where('id', $id)->first();;
        if (!$row) {
            return response()->json([
                'status' => 'error',
                'message' => 'Legend entry not found.'
            ], 404);
        }
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        DB::table('spatial.legend')->where('id', $id)->delete();;
        return response()->json([
            'status' => 'success',
            'message' => 'Legend entry deleted successfully.'
        ]);
    }
}
