<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Document;
use Illuminate\Support\Facades\Storage;
/**
 * @OA\Tag(
 *     name="Documents",
 *     description="Endpoints related to document management, such as creating, listing, and deleting documents."
 * )
*/

class DocumentController extends Controller
{
    /**
     * @OA\Get(
     *     path="/api/documents/public",
     *     summary="Get a list of public documents",
     *     tags={"Documents"},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function public(Request $request)
    {
        $user = Auth::user();
        $role='user_role_viewer';
        //Search user role
        $data = DB::table('tabular.documents')
        ->join('codelist.user_roles', 'user_roles.id', '=', 'documents.role_id')
        ->select('tabular.documents.*')
        ->where('user_roles.name', 'user_role_viewer');
        if($request->project_id) {
            $data->where('project_id', $request->project_id);
        }
        $data->where('type', 'application/pdf');
        $data = $data->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }


    /**
     * @OA\Get(
     *     path="/api/pictures",
     *     summary="Get a list of pictures",
     *     tags={"Documents"},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function pictures(Request $request)
    {
        $user = Auth::user();
        //Search user role
        $data = DB::table('tabular.documents')
        ->leftJoin('codelist.project_locations', 'codelist.project_locations.id', '=', 'tabular.documents.location_id')
        ->leftJoin('tabular.organizations', 'tabular.organizations.id', '=', 'tabular.documents.author_organization_id')
        ->select('tabular.documents.*', 'codelist.project_locations.name AS location', 'tabular.organizations.name AS author_organization');
        if($request->project_id) {
            $data->where('project_id', $request->project_id);
        }
        $data->where('type', '!=', 'application/pdf');
        $data->orderBy("publication_date", "DESC");
        $data = $data->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }


    /**
     * @OA\Post(
     *     path="/api/picture",
     *     tags={"Documents"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new picture",
     *     @OA\RequestBody(
     *          required=true,
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                    required={"name", "path"},
     *                    @OA\Property(property="name", type="string", description="Related activity name"),
     *                    @OA\Property(
     *                     property="path",
     *                     description="Picture file to upload",
     *                     type="string",
     *                     format="binary"
     *                    ),
     *                    @OA\Property(property="description", type="string", description="Summary or description"),
     *                    @OA\Property(property="publication_date", type="date", description="Publication date"),
     *                    @OA\Property(property="author", type="string", description="Picture's author"),
     *                    @OA\Property(property="field", type="string", description="Picture's place or location"),
     *                    @OA\Property(property="location_id", type="string", description="Location ID"),
     *                    @OA\Property(property="project_id", type="string", description="Project ID")
     *                 )
     *            )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Document created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="path", type="string"),
     *                 @OA\Property(property="description", type="text"),
     *                 @OA\Property(property="author", type="text"),
     *                 @OA\Property(property="publication_date", type="date"),
     *                 @OA\Property(property="field", type="text"),
     *                 @OA\Property(property="location_id", type="string"),
     *                 @OA\Property(property="project_id", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    private function createThumbnail($sourcePath, $destinationPath, $fixedHeight, $quality = 90) {
        // Get image details
        list($originalWidth, $originalHeight, $imageType) = getimagesize($sourcePath);
        // Determine image type and create image resource
        switch ($imageType) {
            case IMAGETYPE_JPEG:
                $sourceImage = imagecreatefromjpeg($sourcePath);
                break;
            case IMAGETYPE_PNG:
                $sourceImage = imagecreatefrompng($sourcePath);
                break;
            case IMAGETYPE_GIF:
                $sourceImage = imagecreatefromgif($sourcePath);
                break;
            default:
                echo "Not supported image type";
                return false; // Unsupported image type
        }
        // Calculate new width to maintain aspect ratio
        $aspectRatio = $originalWidth / $originalHeight;
        $newWidth = $fixedHeight * $aspectRatio;
        // Create a new true-color image for the thumbnail
        $thumbnailImage = imagecreatetruecolor($newWidth, $fixedHeight);
        // Copy and resample the original image to the thumbnail
        imagecopyresampled($thumbnailImage, $sourceImage, 0, 0, 0, 0, $newWidth, $fixedHeight, $originalWidth, $originalHeight);
        // Save the thumbnail based on its type
        switch ($imageType) {
            case IMAGETYPE_JPEG:
                imagejpeg($thumbnailImage, $destinationPath, $quality);
                break;
            case IMAGETYPE_PNG:
                imagepng($thumbnailImage, $destinationPath);
                break;
            case IMAGETYPE_GIF:
                imagegif($thumbnailImage, $destinationPath);
                break;
        }
        // Destroy image resources to free up memory
        imagedestroy($sourceImage);
        imagedestroy($thumbnailImage);
        return true;
    }

    public function storePicture(Request $request)
    {
        ini_set('memory_limit', '512M');
        $request->validate([
            'name' => 'required|string',
            'path' => 'required|file|mimetypes:image/jpeg,image/png,image/jpg,image/gif|max:51200',
            'project_id' => 'required|string',
            'author' => 'string',
            'publication_date' => 'date', // Format Y-m-d, etc.
        ]);
        $path = null;
        $fileSize=0;
        $file_type=null;
        if ($request->hasFile('path')) {
            $baseDocumentName = Str::uuid()->toString();
            $file = $request->file('path');
            $file_type = $file->getMimeType();
            $documentName = $baseDocumentName.'.'.$file->extension();
            $path = $file->storeAs('documents', $documentName, 'public');
            $fullPath = Storage::disk('public')->path('documents/' . $documentName);
            //Retrieve image size
            $fileSize = $file->getSize();
            $thumbnailPath = Storage::disk('public')->path('documents/' . $baseDocumentName .'_thumbnail.jpg');
            //Extract image thumbnail
            $this->createThumbnail($fullPath, $thumbnailPath, 240);
        }
        // Check if file has been uploaded
        if (!$path || !$file_type) {
            return response()->json([
                "message" => !$path ? "Document failed to upload." : "Document MIME type can't been extracted",
            ], 500);
        }
        $user = Auth::user();
        $data = Document::create([
            'name' => $request->name,
            'description' => $request->description,
            'size' => $fileSize,
            'type' => $file_type,
            'publication_date' => $request->publication_date,
            'author' => $request->author,
            'field' => $request->field,
            'location_id' => $request->location_id,
            'project_id' => $request->project_id,
            'role_id' => '5b87e4e9-a7a2-4ad5-9218-403fb4b7fccf',
            'user_id' => $user->user_id,
            'path' => $path,
        ]);
        return response()->json([
            'message' => 'Picture created successfully',
            'data' => $data,
        ], 200);
    }



    /**
     * @OA\Get(
     *     path="/api/documents",
     *     summary="Get a list of documents when user is connected",
     *     tags={"Documents"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Response(response=200, description="Successful operation"),
     *     @OA\Response(response=400, description="Invalid request")
     * )
     */
    public function index()
    {
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        $role = $user_role->name;
        //Search user role
        $data = DB::table('tabular.documents')
        ->join('codelist.user_roles', 'user_roles.id', '=', 'documents.role_id')
        ->select('tabular.documents.*');
        if($role==='user_role_viewer') {
            $data->where('user_roles.name', 'user_role_viewer');
        }else if($role==='user_role_editor') {
            $data->whereIn('user_roles.name', ['user_role_editor', 'user_role_viewer']);
        }
        $data->where('type', 'application/pdf');
        /*$sql = $data->toSql(); // Shows SQL with placeholders
        $bindings = $data->getBindings(); // The values to bind
        dd($sql, $bindings);*/
        $data = $data->get();
        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200); // 200 = OK
    }

    /**
     * @OA\Post(
     *     path="/api/documents",
     *     tags={"Documents"},
     *     security={{"bearerAuth":{}}},
     *     summary="Create a new document",
     *     @OA\RequestBody(
     *          required=true,
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                    required={"name", "path"},
     *                    @OA\Property(property="name", type="string", example="Document name"),
     *                    @OA\Property(
     *                     property="path",
     *                     description="Document file to upload",
     *                     type="string",
     *                     format="binary"
     *                    ),
     *                    @OA\Property(property="description", type="text", example="Document description"),
     *                    @OA\Property(property="publication_date", type="date"),
     *                    @OA\Property(property="author", type="string"),
     *                    @OA\Property(property="edition_year", type="integer", example=2024),
     *                    @OA\Property(property="project_id", type="string"),
     *                    @OA\Property(property="role_id", type="string")
     *                 )
     *            )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Document created",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="path", type="string"),
     *                 @OA\Property(property="description", type="text"),
     *                 @OA\Property(property="author", type="text"),
     *                 @OA\Property(property="publication_date", type="date"),
     *                 @OA\Property(property="edition_year", type="integer"),
     * *               @OA\Property(property="project_id", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=401, description="Validation failed")
     * )
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'path' => 'required|file|mimetypes:application/pdf|max:51200',
            'project_id' => 'required|string',
            'role_id' => 'required|string',
            'author' => 'string',
            'edition_year' => [
                'string',
                'regex:/^\d{4}$/',             // Exactly 4 digits
                'digits:4',                    // Laravel rule ensures length 4 digits
                'in:' . implode(',', range(1900, date('Y'))), // optional: limit to valid years
            ],
            'publication_date' => 'date', // Format Y-m-d, etc.
        ]);
        $path = null;
        $fileSize=0;
        $file_type=null;
        if ($request->hasFile('path')) {
            $baseDocumentName = Str::uuid()->toString();
            $file = $request->file('path');
            $file_type = $file->getMimeType();
            $documentName = $baseDocumentName.'.'.$file->extension();
            $path = $file->storeAs('documents', $documentName, 'public');
            $fullPath = Storage::disk('public')->path('documents/' . $documentName);
            //Retrieve image size
            $fileSize = $file->getSize();
            //Extract first document page as image thumbnail
            $outputPath = Storage::disk('public')->path('previews/' . $baseDocumentName .'.jpg');
            $cmd = "gswin64c -dNOPAUSE -dBATCH -sDEVICE=pngalpha -r144 -dFirstPage=1 -dLastPage=1 -sOutputFile=\"$outputPath\" \"$fullPath\"";
            exec($cmd, $out, $code);
            $isPreviewGenerated = $code === 0 && file_exists($outputPath);
        }
        // Check if file has been uploaded
        if (!$path || !$file_type) {
            return response()->json([
                "message" => !$path ? "Document failed to upload." : "Document MIME type can't been extracted",
            ], 500);
        }
        $user = Auth::user();
        $data = Document::create([
            'name' => $request->name,
            'description' => $request->description,
            'size' => $fileSize,
            'type' => $file_type,
            'publication_date' => $request->publication_date,
            'author' => $request->author,
            'edition_year' => $request->edition_year,
            'project_id' => $request->project_id,
            'role_id' => $request->role_id,
            'user_id' => $user->user_id,
            'path' => $path,
        ]);
        return response()->json([
            'message' => 'Document created successfully',
            'is_preview_generated' => $isPreviewGenerated,
            'data' => $data,
        ], 200);
    }

    /**
     * @OA\Delete(
     *     path="/api/document/{id}",
     *     summary="Delete a document",
     *     tags={"Documents"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Document ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Document deleted successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="string", example="success"),
     *             @OA\Property(property="message", type="string", example="Document deleted successfully.")
     *         )
     *     ),
     *     @OA\Response(response=404, description="Document not found")
     * )
     */
    public function delete($id)
    {
        $data = Document::find($id);
        //Check if user has role to delete this document
        if (!$data) {
            return response()->json([
                'status' => 'error',
                'message' => 'Document not found.'
            ], 404);
        }
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        $data->delete();
        return response()->json([
            'status' => 'success',
            'message' => 'Document deleted successfully.'
        ]);
    }

    /**
     * @OA\Patch(
     *     path="/api/document/{id}",
     *     summary="Update an existing document",
     *     tags={"Documents"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Document ID",
     *         @OA\Schema(type="string", format="uuid")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name", "name"},
     *             @OA\Property(property="name", type="string", example="Document name"),
     *                    @OA\Property(property="description", type="text", example="Document description"),
     *                    @OA\Property(property="publication_date", type="date"),
     *                    @OA\Property(property="author", type="string"),
     *                    @OA\Property(property="edition_year", type="integer", example=2024),
     *                    @OA\Property(property="project_id", type="string"),
     *                    @OA\Property(property="role_id", type="string")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Document updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="string"),
     *                 @OA\Property(property="name", type="string"),
     *                 @OA\Property(property="path", type="string"),
     *                 @OA\Property(property="description", type="text"),
     *                 @OA\Property(property="author", type="text"),
     *                 @OA\Property(property="publication_date", type="date"),
     *                 @OA\Property(property="edition_year", type="integer"),
     * *               @OA\Property(property="project_id", type="string"),
     *                 @OA\Property(property="created_at", type="string"),
     *                 @OA\Property(property="updated_at", type="string")
     *             )
     *         )
     *     ),
     *     @OA\Response(response=404, description="Document not found"),
     *     @OA\Response(response=401, description="Validation error")
     * )
     */
    public function update(Request $request, $id)
    {
        // Validate only present fields
        $validated = $request->validate([
            'name' => 'string',
            'project_id' => 'string',
            'location_id' => 'string',
            'role_id' => 'string',
            'field' => 'string',
            'description' => 'string',
            'author' => 'string',
            'edition_year' => [
                'string',
                'regex:/^\d{4}$/',             // Exactly 4 digits
                'digits:4',                    // Laravel rule ensures length 4 digits
                'in:' . implode(',', range(1900, date('Y'))), // optional: limit to valid years
            ],
            'publication_date' => 'date', // Format Y-m-d, etc.
        ]);
        //Check if user is administrator
        $user = Auth::user();
        $user_role = DB::table('codelist.user_roles')
        ->select('codelist.user_roles.*')
        ->where('id', $user->role_id)->first();
        if ($user_role->name!=='user_role_administrator') {
            return response()->json([
                "message" => "Authorization denied",
            ], 500);
        }
        // Find the document or return 404
        $data = Document::findOrFail($id);
        // Update document with validated data
        $data->update($validated);
        return response()->json([
            'status' => 'success',
            'message' => 'Document updated successfully.',
            'data' => $data
        ]);
    }
}
